// --------------------------------------------------------------------------------
//   History
// --------------------------------------------------------------------------------
// 08-01-2024 : first version, compiles without errors
//              This sketch receives commands via the USB connection to control
//              the 10 MCP23S17 2x8bit I/O expander ICs. The 10 ICs are all only
//              used as output ports, thus offering 160 digital outputs.
//              144 digital outputs are used to drive the 144 LEDs of the RK11
//              front panel.
//              The output hardware consists of two PCBs, each with 5 MCP23S17
//              devices plus ULN2803 output buffers. As one PCB drives 72 LEDs,
//              each PCB has 8 digital outputs that can be used for other purposes.
//              A command consists of 2 bytes. The first byte defined the 8-bit
//              port to be written, the second byte is the data to be written.
// ===================================================================================

#include  <SPI.h>                         // Arduino SPI library
#include  <MCP23S17.h>                    // Majenko MCP23S17 library
                                          // https://github.com/MajenkoLibraries/MCP23S17


const  uint8_t   leftSelect  = 9;         // MCP23S17 chip select LEFT PCB
const  uint8_t   rightSelect = 8;         // MCP23S17 chip select RIGHT PCB

const  uint8_t   portA       = 0;         // MCP23S17 port A identifier
const  uint8_t   portB       = 1;         // MCP23S17 port B identifier
const  uint8_t   LEFT        = 0;         // board identification
const  uint8_t   RIGHT       = 1;


#define   ASCII_MODE                      // if defined: command in ASCII data
                                          // if not defined: command in binary mode

                                          // instantiate the MCP23S17 devices
MCP23S17  Left1(&SPI, leftSelect, 0);     // at address 0
MCP23S17  Left2(&SPI, leftSelect, 1);     // at address 1
MCP23S17  Left3(&SPI, leftSelect, 2);     // at address 2
MCP23S17  Left4(&SPI, leftSelect, 3);     // at address 3
MCP23S17  Left5(&SPI, leftSelect, 4);     // at address 4

MCP23S17  Right1(&SPI, rightSelect, 0);   // at address 0
MCP23S17  Right2(&SPI, rightSelect, 1);   // at address 1
MCP23S17  Right3(&SPI, rightSelect, 2);   // at address 2
MCP23S17  Right4(&SPI, rightSelect, 3);   // at address 3
MCP23S17  Right5(&SPI, rightSelect, 4);   // at address 4


#define       COMMAND_SIZE                2                    // command is 2 bytes
#define       COMMAND_BUFFER              100 * COMMAND_SIZE   // # commands in buffer

uint8_t       commandBuffer[COMMAND_BUFFER];       // received command queue
unsigned int  cmdHead;                             // command insert point in queue
unsigned int  cmdTail;                             // command retrieve point in queue

uint8_t       cmd1stByte;                          // 1st coomand byte
uint8_t       cmd2ndByte;                          // 2nd commandd byte
uint8_t       asciiByte;                           // received (ASCII) character
uint16_t      dataByte;                            // aggragated received (ASCII) value

unsigned int  receiveState;                        // command receiver state machine
uint8_t       portNumber;                          // port number (0..19)


#define       RX_IDENTIFIER               0        // state: await 1st command byte
#define       RX_DATAVALUE                1        // state: await data command byte
#define       RX_DATAVALUE2               2        // state: await 2nd ASCII data command byte
#define       RX_TERMINATOR               3        // state: await command terminator
#define       RX_IN_SYNC                  4        // state: recovering from error

uint8_t       command[COMMAND_SIZE];               // command to process
uint8_t       shadow[20];                          // "shadow" state of the output ports



// ---------------------------------------------------------------------------------------
// initCommandQueue - initialize the command queue administration.
//
//   This function is called once from setup()
//
//   Input parameters  : -
//   Output parameters : -
//
//   Functional description.
//      The command queue administration indices and state machine are initialized.
// ---------------------------------------------------------------------------------------
void initCommandQueue() {
    cmdHead = 0;                    // command queue insert index
    cmdTail = 0;                    // command queue retrieve index
    receiveState = RX_IDENTIFIER;   // state machine
}




// ---------------------------------------------------------------------------------------
// storeCommand - store received command in queue
//
//   This function is called from receiveCommand().
//
//   Input parameters  : portNumber, cmd2ndByte - received command.
//   Output parameters : -
//
//   Functional description.
//      The input parameters are stored in a queue (commandBuffer).
// ---------------------------------------------------------------------------------------
void storeCommand() {
  unsigned int nextIndex;

    nextIndex = cmdHead + COMMAND_SIZE;
    if (nextIndex >= COMMAND_BUFFER) {
        nextIndex = 0;
    }
    if (nextIndex != cmdTail) {
       // space available in queue: store the new command.
       // If no space is available, the command is lost!
       commandBuffer[cmdHead]   = portNumber;               // store command in queue
       commandBuffer[cmdHead+1] = cmd2ndByte;
       cmdHead = nextIndex;
    }
}




// ---------------------------------------------------------------------------------------
// receiveCommand - receive a complete command
//
//   This function is called from main().
//
//   Input parameters  : -
//   Output parameters : -
//                       received command stored in commandBuffer (if space available)
//
//   Functional description.
//      This routine is called every main loop cycle.
//      When the first byte of a command is received, it is validated and stored.
//      When the second bye of a command is received, the command is stored in the queue.
//
//   Command description
//   -------------------
//   The command can be sent in "binary" mode or in "ASCII" mode.
//
//   In both modes, the first byte contains the destination for the data.
//   The destination can be either the LEFT board or the RIGHT board of the hardware.
//   Further, on each board are five MCP23S17 ICs, and each IC has two 8-bit ports.
//   The first command byte encodes the board (LEFT/RIGHT) and which 8-bit port.
//   For the LEFT board the "base" command is the letter "L" and this is "incremented"
//   with the IC number. The valid command letters for the LEFT board are "L,M,N,O,P".
//   These letters indicate that the data is for port A of the IC. The letter is changed
//   to lowercase to indicate that the data is for port B of the IC.
//   For the RIGHT board, the "base" command letter is "R". Thus, the valid letters are
//   "R,S,T,U,V" for the ICs and port A, and "r,s,t,u,v" for port B.
//
//   Special command letter is "X". Whatever follows is don't care, but upon termination,
//   the RK11-C interface is initialized. All outputs are set to zero and the shadow memory
//   is initialized to 0.
//
//   In binary mode, the second command byte is the port data (hexadecimal, one byte).
//   In ASCII mode, two ASCII hex numbers (0..9, A..F or a..f) follow, representing the data.
//
//   Each command is terminated with the CR <carriage return> character (hex 0D).
// ---------------------------------------------------------------------------------------

void receiveCommand() {
    switch (receiveState) {
        case RX_IDENTIFIER: {
            if (Serial.available() > 0) {
                cmd1stByte = Serial.read();
                receiveState = RX_DATAVALUE;      // assume that the first token is valid
                // validate command identifier
                switch (cmd1stByte) {
                    case 'L' : { portNumber = 0;  break; }
                    case 'M' : { portNumber = 2;  break; }
                    case 'N' : { portNumber = 4;  break; }
                    case 'O' : { portNumber = 6;  break; }
                    case 'P' : { portNumber = 8;  break; }
                    case 'l' : { portNumber = 1;  break; }
                    case 'm' : { portNumber = 3;  break; }
                    case 'n' : { portNumber = 5;  break; }
                    case 'o' : { portNumber = 7;  break; }
                    case 'p' : { portNumber = 9;  break; }
                    //
                    case 'R' : { portNumber = 10;  break; }
                    case 'S' : { portNumber = 12;  break; }
                    case 'T' : { portNumber = 14;  break; }
                    case 'U' : { portNumber = 16;  break; }
                    case 'V' : { portNumber = 18;  break; }
                    case 'r' : { portNumber = 11;  break; }
                    case 's' : { portNumber = 13;  break; }
                    case 't' : { portNumber = 15;  break; }
                    case 'u' : { portNumber = 17;  break; }
                    case 'v' : { portNumber = 19;  break; }
                    //
                    case 'X' : { portNumber = 20;  break; }  // initialize

                    default  : {
                        // not valid: reject, await termination
                        receiveState = RX_TERMINATOR;
                        break;
                    }
                }
            }
            break;
        }
        case RX_DATAVALUE: {
            if (Serial.available() > 0) {
                asciiByte = Serial.read();

                #ifdef ASCII_MODE
                    // first digit *must* be an ASCII hex number (0..9, A..F, a..f)
                    if ( ( (asciiByte >= 0x30) & (asciiByte <= 0x39) ) |
                         ( (asciiByte >= 0x41) & (asciiByte <= 0x46) ) |
                         ( (asciiByte >= 0x61) & (asciiByte <= 0x66) )   ) {
                        // valid hex number (0..9, A..F, a..f): convert to binary
                        if (asciiByte > 0x46) {
                            asciiByte = asciiByte & 0xDF;      // force uppercase
                        }
                        asciiByte = asciiByte - 0x30;
                        if (asciiByte > 9) {
                            asciiByte = asciiByte - 7;
                        }
                        dataByte = (uint16_t)(asciiByte);
                        cmd2ndByte = (uint8_t)dataByte;
                        receiveState = RX_DATAVALUE2;
                    }
                    else {
                        // not valid: reject, await termination
                        receiveState = RX_TERMINATOR;
                    }

                #else
                    cmd2ndByte = asciiByte;
                    receiveState = RX_TERMINATOR;
                #endif
            }
            break;
        }
        case RX_DATAVALUE2: {
            if (Serial.available() > 0) {
                asciiByte = Serial.read();
                
                    // second digit *must* be an ASCII hex number (0..9, A..F, a..f)
                    if ( ( (asciiByte >= 0x30) & (asciiByte <= 0x39) ) |
                         ( (asciiByte >= 0x41) & (asciiByte <= 0x46) ) |
                         ( (asciiByte >= 0x61) & (asciiByte <= 0x66) )   ) {
                        // valid hex number (0..9, A..F, a..f): convert to binary
                        if (asciiByte > 0x46) {
                            asciiByte = asciiByte & 0xDF;      // force uppercase
                        }
                        asciiByte = asciiByte - 0x30;
                        if (asciiByte > 9) {
                            asciiByte = asciiByte - 7;
                        }
                        dataByte = dataByte << 4;
                        dataByte = dataByte + (uint16_t)(asciiByte);
                        cmd2ndByte = (uint8_t)dataByte;
                        receiveState = RX_TERMINATOR;
                    }
                    else {
                        // not valid: reject, await termination
                        receiveState = RX_TERMINATOR;
                    }
                }
            break;
        }
        case RX_TERMINATOR: {
            if (Serial.available() > 0) {
                if (Serial.read() == 0x0D) {
                    storeCommand();
                    receiveState = RX_IDENTIFIER;
                }
                else {
                    // bad termination, await termination to return "in sync"
                    receiveState = RX_IN_SYNC;
                }
            }
            break;
        }
        case RX_IN_SYNC: {
            if (Serial.available() > 0) {
                if (Serial.read() == 0x0D) {
                    receiveState = RX_IDENTIFIER;
                }
            }
            break;
        }
    }
}




// ---------------------------------------------------------------------------------------
// retrieveCommand - retrieve a command from the queue (if available)
//
//   This function is called from the main loop.
//
//   Input parameters  : -
//   Output parameters : command[] - the command to be processed.
//   Return value      : false - if queue is empty
//                       true  - if a command is available, loaded in command[].
//
//   Functional description.
//      A command is retrieved from the queue (if the queue is not empty),
//      and stored in command[].
// ---------------------------------------------------------------------------------------
boolean retrieveCommand() {
  unsigned int nextIndex;

    if (cmdHead != cmdTail) {
        command[0] = commandBuffer[cmdTail];         // retrieve command from queue
        command[1] = commandBuffer[cmdTail+1];       // and store in "local" command
        nextIndex = cmdTail + COMMAND_SIZE;
        if (nextIndex >= COMMAND_BUFFER) {
            nextIndex = 0;
        }
        cmdTail = nextIndex;
        return true;
    } else {
        // queue is empty
        return false;
    }
}




// ################################################################################
// ################################################################################


void allPortsOff() {
    Left1.writePort(portA, 0);
    Left1.writePort(portB, 0);
    Left2.writePort(portA, 0);
    Left2.writePort(portB, 0);
    Left3.writePort(portA, 0);
    Left3.writePort(portB, 0);
    Left4.writePort(portA, 0);
    Left4.writePort(portB, 0);
    Left5.writePort(portA, 0);
    Left5.writePort(portB, 0);

    Right1.writePort(portA, 0);
    Right1.writePort(portB, 0);
    Right2.writePort(portA, 0);
    Right2.writePort(portB, 0);
    Right3.writePort(portA, 0);
    Right3.writePort(portB, 0);
    Right4.writePort(portA, 0);
    Right4.writePort(portB, 0);
    Right5.writePort(portA, 0);
    Right5.writePort(portB, 0);

    // initialize "shadow memory" (to avoid needless write actions)
    for (unsigned int ix = 0;  ix < 20;  ix++)
        shadow[ix] = 0;
}



void setup() {
  uint8_t pin;

    Serial.begin(115200);
    initCommandQueue();
    
    // initialize SPI devices and define all pins as output

    Left1.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Left1.pinMode(pin, OUTPUT);
    Left2.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Left2.pinMode(pin, OUTPUT);
    Left3.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Left3.pinMode(pin, OUTPUT);
    Left4.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Left4.pinMode(pin, OUTPUT);
    Left5.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Left5.pinMode(pin, OUTPUT);

    Right1.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Right1.pinMode(pin, OUTPUT);
    Right2.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Right2.pinMode(pin, OUTPUT);
    Right3.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Right3.pinMode(pin, OUTPUT);
    Right4.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Right4.pinMode(pin, OUTPUT);
    Right5.begin();
    for ( pin = 0;  pin < 16;  pin++)
        Right5.pinMode(pin, OUTPUT);

    allPortsOff();   // set all outputs to "OFF"
}


// ################################################################################
// ################################################################################


void loop() {
  unsigned int ix;
  uint8_t      data;

    receiveCommand();   // receive commands and store in command queue

    if (retrieveCommand() == true) {
        // command available from queue: process the command
        // first check whether an actual write to the output port is needed
        ix = command[0];
        data = command[1];

        if (ix == 20) {
            // special case: command "X" - initialize interface
            allPortsOff();
        }
        else {
            if ( shadow[ix] != data ) {
                // update required
                shadow[ix] = data;

                switch (ix) {
                    case  0: { Left1.writePort(portA, data);   break; }
                    case  1: { Left1.writePort(portB, data);   break; }
                    case  2: { Left2.writePort(portA, data);   break; }
                    case  3: { Left2.writePort(portB, data);   break; }
                    case  4: { Left3.writePort(portA, data);   break; }
                    case  5: { Left3.writePort(portB, data);   break; }
                    case  6: { Left4.writePort(portA, data);   break; }
                    case  7: { Left4.writePort(portB, data);   break; }
                    case  8: { Left5.writePort(portA, data);   break; }
                    case  9: { Left5.writePort(portB, data);   break; }

                    case 10: { Right1.writePort(portA, data);  break; }
                    case 11: { Right1.writePort(portB, data);  break; }
                    case 12: { Right2.writePort(portA, data);  break; }
                    case 13: { Right2.writePort(portB, data);  break; }
                    case 14: { Right3.writePort(portA, data);  break; }
                    case 15: { Right3.writePort(portB, data);  break; }
                    case 16: { Right4.writePort(portA, data);  break; }
                    case 17: { Right4.writePort(portB, data);  break; }
                    case 18: { Right5.writePort(portA, data);  break; }
                    case 19: { Right5.writePort(portB, data);  break; }
                }
            }
        }
    }
}
