import wx
import serial
import locale
from wx.lib.buttons import GenBitmapButton

################################################################# written/debugged 30-JAN-2024 ####
class TabPanel(wx.Panel):
    def __init__(self, parent):
        wx.Panel.__init__(self, parent=parent, id=wx.ID_ANY)

        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.SetSizer(sizer)

        wx.StaticLine(self, -1, (5, 335), (1150, 3))
###################################################################################################


class NotebookDemo(wx.Notebook):
    def __init__(self, parent):
        self.locale = wx.Locale(wx.LANGUAGE_ENGLISH)
        wx.Notebook.__init__(self, parent, id=wx.ID_ANY, style=wx.BK_TOP)

        self.commPortID          = ""
        self.previousCommPort    = ""
        self.commPortIDtoClose   = ""
        self.commPortOpened      = False         # flag COMM port opened
        self.commPortValid       = False         # only for "error sign" blinking
        self.errorCommPortRate   = 30            # flash rate interval [*10 ms]
        self.ComPort             = 0             # make sure it is defined
        self.baudrate            = 115200        # default baudrate = 115200 Bd

        self.activeTab           = 0             # active tab when application starts
        self.rowDistance         = 65            # distance between LED rows

        # load bitmap images
        self.errorImage          = wx.Bitmap("error.gif")
        self.digitalLogo         = wx.Bitmap("digital-logo.gif")
        self.ledOnImage          = wx.Bitmap("led-on-white.gif")    # white LEDs panel
        # self.ledOnImage        = wx.Bitmap("led-on-red.gif")      # red LEDs panel
        self.ledOffImage         = wx.Bitmap("led-off.gif")

        #define used fonts
        self.smallFont           = wx.Font(8, wx.DEFAULT, wx.NORMAL, wx.NORMAL)
        self.labelFont           = wx.Font(8, wx.FONTFAMILY_TELETYPE, wx.NORMAL, wx.BOLD)
        self.groupFont           = wx.Font(12, wx.FONTFAMILY_TELETYPE, wx.NORMAL, wx.BOLD)
        

    # The LED images are stored in the ledOnList[] and the ledOffList[].
    # The range index for both list is 0..143.
    # Eight LEDs are connected to an output port of the MCP23S17 so all 8 LEDs are written.
    # The state of all LEDs (on or off) must be administrated, so that the update of one LED
    # does not affect the other 7 LEDs connected to that output port.
    # Which 8 LEDs are connected to one output port depends on the board (LEFT or RIGHT).
    # Based on the index (0..143) of the LED number, the command letter and bitposition is
    # looked up in the commandLetter[] and commandBit[] lists. Numbering of the LEDs goes
    # from top left corner is one, top right corner is 36, etc. Next row starts at the left
    # with LED number 37 etc. The LED number of the bottom right LED is 143.
    # NOTE: each board has five MCP23S27 ICs, so there are actually 80 outputs on each board.
    #       On each board are two 8-bit ports of which only 4 bits are used for the LEDs, the
    #       other 4 bits are connected to the EXP1 and EXP2 header ("spare outputs").
    #       This test app does not control the spare outputs, they are held at 0.

        self.commandLetter  = [ 'L','L','L','L','L','L','L','L','M','M','M','M','M','M','M','M','p','p',
                                'R','R','R','R','R','R','R','R','S','S','S','S','S','S','S','S','v','v',
                                
                                'm','m','m','l','l','l','l','l','l','l','l','m','m','m','m','m','p','p',
                                'r','r','r','r','r','r','r','r','s','s','s','s','s','s','s','s','v','v',
                                
                                'N','N','n','n','n','n','n','n','n','n','o','o','o','o','o','o','o','o',
                                't','t','t','t','t','t','t','t','T','T','u','u','u','u','u','u','u','u',
                                
                                'N','N','N','N','N','N','O','O','O','O','O','O','O','O','P','P','P','P',
                                'T','T','T','T','T','T','U','U','U','U','U','U','U','U','V','V','V','V'  ]

        self.commandBit = [ 0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,
                            0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,
                            
                            0x80,0x40,0x20,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x10,0x08,0x04,0x02,0x01,0x20,0x10,
                            0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x20,0x10,
                            
                            0x80,0x40,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,
                            0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80,0x01,0x02,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,
                            
                            0x20,0x10,0x08,0x04,0x02,0x01,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x08,0x04,0x02,0x01,
                            0x80,0x40,0x20,0x10,0x08,0x04,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01,0x08,0x04,0x02,0x01  ]

        # "shadow" list of current LED state (port bytes), initially all "OFF"
        self.state = [0] * 20

        # command string
        self.sendCmdString  = ""

        # Exit button
        bottomYPos = 368
        self.buttonExit = wx.Button(self, label="Exit", pos=(1040, bottomYPos+2), size=(100, 20))
        self.buttonExit.SetForegroundColour('#ff0000')
        self.buttonExit.Bind(wx.EVT_BUTTON, self.ExitClick)

        # Clear All button
        self.buttonClear = wx.Button(self, label="Clear all", pos=(840, bottomYPos+2), size=(100, 20))
        self.buttonClear.SetForegroundColour('#0000ff')
        self.buttonClear.Bind(wx.EVT_BUTTON, self.ClearAllClick)

        # DIGITAL logo and application identification
        self.digitalLogoImage = wx.StaticBitmap(self, -1, self.digitalLogo)
        self.digitalLogoImage.SetPosition((16, bottomYPos-2))
        self.IDline = wx.StaticText(self, pos=(120, bottomYPos+8))
        self.IDline.SetLabel('RK11-C  Indicator Panel  - V0.10  Jan 2024.')
        self.IDline.SetFont(self.smallFont)
        self.IDline.SetForegroundColour("#0000FF")

        # define timer
        self.timer = wx.Timer(self, -1)
        self.Bind(wx.EVT_TIMER, self.OnTimer)
        self.timerIsRunning = False
        self.startTimer(500)                  # blink rate, COM port initially not assigned



#==================================================================================================
#-- TAB #1  Connection  ---------------------------------------------------------------------------
#==================================================================================================
        tabOne = TabPanel(self)
        tabOne.SetBackgroundColour("White")
        self.AddPage(tabOne, " RK11-C serial connection  ")

        # find the available COM ports (code works only for Windows)
        COMports = ['COM%s' % (i + 1) for i in range(64)]
        availableCOMports = []
        for port in COMports:
            try:
                s = serial.Serial(port)
                s.close()
                availableCOMports.append(port)
            except (OSError, serial.SerialException):
                pass
        print("--- discovered COM ports:", availableCOMports)

        # display COM port selection
        comPortXpos = 60
        comPortYpos = 40
        wx.StaticBox(tabOne, label=' COM port ', pos=(comPortXpos, comPortYpos), size=(140, 96))
        self.cbComm1 = wx.ComboBox(tabOne, pos=(comPortXpos+14, comPortYpos+27), size=(70, 25),
                                   choices=availableCOMports, style=wx.CB_READONLY)
        self.cbComm1.SetToolTip(wx.ToolTip("set COM port"))
        self.cbComm1.Bind(wx.EVT_COMBOBOX, self.OnSelectCommPort)
        # flashing "no COM port selected" warning
        self.errorCommPort = wx.StaticBitmap(tabOne, -1, self.errorImage, pos=(comPortXpos+98, comPortYpos+25))
        self.errorCommPortShown = True
        # COM port OPEN/CLOSE button
        self.buttonCommOpen = wx.Button(tabOne, label="", pos=(comPortXpos+13, comPortYpos+60), size=(114, 23))
        self.buttonCommOpen.SetBackgroundColour('#CCCCCC')
        self.buttonCommOpen.SetForegroundColour('#000000')
        self.buttonCommOpen.Bind(wx.EVT_BUTTON, self.onCommOpenClick)



#==================================================================================================
#-- TAB #2  RK11-C Panel control  -----------------------------------------------------------------
#==================================================================================================
        self.tabTwo = TabPanel(self)
        self.tabTwo.SetBackgroundColour("White")
        self.AddPage(self.tabTwo, " RK11-C control ")

        self.tabTwo.Bind(wx.EVT_PAINT, self.paintPanel)

        self.ledOnList = []
        self.ledOffList = []
        # draw four rows of 36 lights
        yStart = 64
        yOffset = 0
        for row in range(4):
            xStart = 30
            xOffset = 0
            for ix in range(36):
                # create "led ON" instances and hide them
                self.ledOnList.append( GenBitmapButton(self.tabTwo, wx.ID_ANY, bitmap=self.ledOnImage,
                                                       style=wx.NO_BORDER|wx.BU_EXACTFIT,
                                                       pos=(xStart+xOffset, yStart+yOffset),
                                                       size=(30,30), name=str(ix+row*36)) )
                self.ledOnList[ix+row*36].Bind(wx.EVT_LEFT_DOWN, self.onClickLedOn)
                self.ledOnList[ix+row*36].Hide()
                # create "led OFF" instances (visible)
                self.ledOffList.append( GenBitmapButton(self.tabTwo, wx.ID_ANY, bitmap=self.ledOffImage,
                                                        style=wx.NO_BORDER|wx.BU_EXACTFIT,
                                                        pos=(xStart+xOffset, yStart+yOffset),
                                                        size=(30,30), name=str(ix+row*36)) )
                self.ledOffList[ix+row*36].Bind(wx.EVT_LEFT_DOWN, self.onClickLedOff)
                xOffset = xOffset + 30
            yOffset = yOffset + self.rowDistance

        # text legends
        self.row1TextList = []
        self.row2TextList = []
        self.row3TextList = []
        self.row4TextList = []
        xStart = 36
        yStart = 53
        xOffset = 30
        yOffset = self.rowDistance
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="ERR", pos=(xStart+xOffset*0,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="HE",  pos=(xStart+xOffset*1+3,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="SCP", pos=(xStart+xOffset*2,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="MNT", pos=(xStart+xOffset*3,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="IAI", pos=(xStart+xOffset*4,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="FMT", pos=(xStart+xOffset*5,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="RWA", pos=(xStart+xOffset*6,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="SSE", pos=(xStart+xOffset*7,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="CR",  pos=(xStart+xOffset*8+3,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="IDE", pos=(xStart+xOffset*9,  yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*10+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*11+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*12+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*13+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*14+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="GO",  pos=(xStart+xOffset*15+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*16, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*17, yStart+yOffset*0), style=wx.ALIGN_LEFT) )

        self.row1TextList.append( wx.StaticText(self.tabTwo, label="15",  pos=(xStart+xOffset*18+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="14",  pos=(xStart+xOffset*19+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="13",  pos=(xStart+xOffset*20+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="12",  pos=(xStart+xOffset*21+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="11",  pos=(xStart+xOffset*22+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="10",  pos=(xStart+xOffset*23+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="09",  pos=(xStart+xOffset*24+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="08",  pos=(xStart+xOffset*25+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="07",  pos=(xStart+xOffset*26+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="06",  pos=(xStart+xOffset*27+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="05",  pos=(xStart+xOffset*28+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="04",  pos=(xStart+xOffset*29+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*30+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*31+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*32+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*33+3, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="LWD", pos=(xStart+xOffset*34, yStart+yOffset*0), style=wx.ALIGN_LEFT) )
        self.row1TextList.append( wx.StaticText(self.tabTwo, label="RDG", pos=(xStart+xOffset*35, yStart+yOffset*0), style=wx.ALIGN_LEFT) )

        self.row2TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*0+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*1+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*2+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="07",  pos=(xStart+xOffset*3+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="06",  pos=(xStart+xOffset*4+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="05",  pos=(xStart+xOffset*5+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="04",  pos=(xStart+xOffset*6+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*7+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*8+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*9+3,  yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*10+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="SRF", pos=(xStart+xOffset*11, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*12+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*13+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*14+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*15+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="PO",  pos=(xStart+xOffset*16+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="HOK", pos=(xStart+xOffset*17, yStart+yOffset*1), style=wx.ALIGN_LEFT) )

        self.row2TextList.append( wx.StaticText(self.tabTwo, label="15",  pos=(xStart+xOffset*18+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="14",  pos=(xStart+xOffset*19+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="13",  pos=(xStart+xOffset*20+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="12",  pos=(xStart+xOffset*21+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="11",  pos=(xStart+xOffset*22+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="10",  pos=(xStart+xOffset*23+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="09",  pos=(xStart+xOffset*24+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="08",  pos=(xStart+xOffset*25+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="07",  pos=(xStart+xOffset*26+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="06",  pos=(xStart+xOffset*27+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="05",  pos=(xStart+xOffset*28+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="04",  pos=(xStart+xOffset*29+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*30+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*31+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*32+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*33+3, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*34, yStart+yOffset*1), style=wx.ALIGN_LEFT) )
        self.row2TextList.append( wx.StaticText(self.tabTwo, label="IDL", pos=(xStart+xOffset*35, yStart+yOffset*1), style=wx.ALIGN_LEFT) )

        self.row3TextList.append( wx.StaticText(self.tabTwo, label="BM",  pos=(xStart+xOffset*0+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*1,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="15",  pos=(xStart+xOffset*2+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="14",  pos=(xStart+xOffset*3+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="13",  pos=(xStart+xOffset*4+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="12",  pos=(xStart+xOffset*5+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="11",  pos=(xStart+xOffset*6+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="10",  pos=(xStart+xOffset*7+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="09",  pos=(xStart+xOffset*8+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="08",  pos=(xStart+xOffset*9+3,  yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="07",  pos=(xStart+xOffset*10+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="06",  pos=(xStart+xOffset*11+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="05",  pos=(xStart+xOffset*12+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="04",  pos=(xStart+xOffset*13+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*14+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*15+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*16+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*17+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )

        self.row3TextList.append( wx.StaticText(self.tabTwo, label="WTG", pos=(xStart+xOffset*18, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="ROH", pos=(xStart+xOffset*19, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="DRE", pos=(xStart+xOffset*20, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="OVR", pos=(xStart+xOffset*21, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="WLO", pos=(xStart+xOffset*22, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="SKE", pos=(xStart+xOffset*23, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="PGE", pos=(xStart+xOffset*24, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="NXM", pos=(xStart+xOffset*25, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="DLT", pos=(xStart+xOffset*26, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="TE",  pos=(xStart+xOffset*27+3, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="NXD", pos=(xStart+xOffset*28, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="NXC", pos=(xStart+xOffset*29, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="NXS", pos=(xStart+xOffset*30, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*31, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*32, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*33, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="CSE", pos=(xStart+xOffset*34, yStart+yOffset*2), style=wx.ALIGN_LEFT) )
        self.row3TextList.append( wx.StaticText(self.tabTwo, label="WCE", pos=(xStart+xOffset*35, yStart+yOffset*2), style=wx.ALIGN_LEFT) )

        self.row4TextList.append( wx.StaticText(self.tabTwo, label="IRQ", pos=(xStart+xOffset*0,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="WCO", pos=(xStart+xOffset*1,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*2+3,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*3+3,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*4+3,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="DPL", pos=(xStart+xOffset*5,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="HD",  pos=(xStart+xOffset*6+3,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="DRU", pos=(xStart+xOffset*7,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="SIN", pos=(xStart+xOffset*8,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="SOK", pos=(xStart+xOffset*9,  yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="DRY", pos=(xStart+xOffset*10, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="RWS", pos=(xStart+xOffset*11, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="WPS", pos=(xStart+xOffset*12, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="SEQ", pos=(xStart+xOffset*13, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*14+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*15+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*16+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*17+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )

        self.row4TextList.append( wx.StaticText(self.tabTwo, label="PRE", pos=(xStart+xOffset*18, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="HDR", pos=(xStart+xOffset*19, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="DAT", pos=(xStart+xOffset*20, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="CHK", pos=(xStart+xOffset*21, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="PA",  pos=(xStart+xOffset*22+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="   ", pos=(xStart+xOffset*23, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="07",  pos=(xStart+xOffset*24+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="06",  pos=(xStart+xOffset*25+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="05",  pos=(xStart+xOffset*26+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="04",  pos=(xStart+xOffset*27+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*28+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*29+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*30+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*31+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="03",  pos=(xStart+xOffset*32+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="02",  pos=(xStart+xOffset*33+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="01",  pos=(xStart+xOffset*34+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )
        self.row4TextList.append( wx.StaticText(self.tabTwo, label="00",  pos=(xStart+xOffset*35+3, yStart+yOffset*3), style=wx.ALIGN_LEFT) )

        # set label font, size, background and text color
        for ix in range(36):
            self.row1TextList[ix].SetFont(self.labelFont)
            self.row1TextList[ix].SetForegroundColour("#FFFFFF")
            self.row1TextList[ix].SetBackgroundColour("#000000")

            self.row2TextList[ix].SetFont(self.labelFont)
            self.row2TextList[ix].SetForegroundColour("#FFFFFF")
            self.row2TextList[ix].SetBackgroundColour("#000000")

            self.row3TextList[ix].SetFont(self.labelFont)
            self.row3TextList[ix].SetForegroundColour("#FFFFFF")
            self.row3TextList[ix].SetBackgroundColour("#000000")

            self.row4TextList[ix].SetFont(self.labelFont)
            self.row4TextList[ix].SetForegroundColour("#FFFFFF")
            self.row4TextList[ix].SetBackgroundColour("#000000")

        # "group" names
        self.groupTextList = []
        groupYpos = 29
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="MEX", pos=(346, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="FUNC", pos=(417, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="DATA BUFFER", pos=(755, groupYpos), style=wx.ALIGN_LEFT) )
        groupYpos = groupYpos + self.rowDistance
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="DRIVE", pos=(52, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="CYLINDER", pos=(201, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="SECTOR", pos=(419, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="DSB", pos=(796, groupYpos), style=wx.ALIGN_LEFT) )
        groupYpos = groupYpos + self.rowDistance
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="BUS ADDRESS", pos=(277, groupYpos), style=wx.ALIGN_LEFT) )
        groupYpos = groupYpos + self.rowDistance
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="ID", pos=(127, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="CSEC", pos=(491, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="IWC", pos=(856, groupYpos), style=wx.ALIGN_LEFT) )
        self.groupTextList.append( wx.StaticText(self.tabTwo, label="BC", pos=(1041, groupYpos), style=wx.ALIGN_LEFT) )

        # set group font, size, background and text color
        for ix in range(12):
            self.groupTextList[ix].SetFont(self.groupFont)
            self.groupTextList[ix].SetForegroundColour("#FFFFFF")
            self.groupTextList[ix].SetBackgroundColour("#000000")



# =================================================================================================
# =================================================================================================
#
#       self.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGED, self.OnPageChanged)
#       self.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGING, self.OnPageChanging)
#
# =================================================================================================
#
#   def OnPageChanged(self, event):
#       old = event.GetOldSelection()
#       new = event.GetSelection()
#       self.activeTab = self.GetSelection()
#       # print('OnPageChanged,  old:%d, new:%d, activeTab:%d\n' % (old, new, self.activeTab))
#       event.Skip()
#
#
#   def OnPageChanging(self, event):
#       old = event.GetOldSelection()
#       new = event.GetSelection()
#       sel = self.GetSelection()
#       print('OnPageChanging:,  old:%d, new:%d, sel:%d\n' % (old, new, sel))
#       event.Skip()
#
# =================================================================================================
# =================================================================================================
# =================================================================================================

    def ExitClick(self, event):
        self.closeCOMMport(self.commPortID)
        print("--- Exit RK11-C Indicator Panel application.")
        frame.Close()


# =================================================================================================
#  COMMUNICATION
# =================================================================================================

    def OnSelectCommPort(self, entry):
        self.previousCommPort = self.commPortID
        self.commPortID = entry.GetString()
        self.commPortValid = True   # stop error blinking
        if (self.previousCommPort == ""):
            # no COM port ever selected yet : initial situation only
            self.buttonCommOpen.SetBackgroundColour('#ffcc00')
            self.buttonCommOpen.SetLabel("Open COM port")
        if (self.previousCommPort == self.commPortID):
            # selection is the same as current selection : no change
            pass
        if (self.previousCommPort != self.commPortID):
            # if previous selected port was opened, it must be closed
            self.closeCOMMport(self.previousCommPort)
            self.buttonCommOpen.SetBackgroundColour('#ffcc00')
            self.buttonCommOpen.SetLabel("Open COM port")

    def onCommOpenClick(self, event):
        if self.commPortID != "":
            if (self.buttonCommOpen.GetLabel() == "Open COM port"):
                self.openCOMMport()
                self.buttonCommOpen.SetLabel("Close COM port")
                self.buttonCommOpen.SetBackgroundColour('#ccff00')
            elif (self.buttonCommOpen.GetLabel() == "Close COM port"):
                self.closeCOMMport(self.commPortID)
                self.buttonCommOpen.SetLabel("Open COM port")
                self.buttonCommOpen.SetBackgroundColour('#ffcc00')

    def openCOMMport(self):
        print(">   opening COM port", self.commPortID)
        self.ComPort = serial.Serial(self.commPortID, self.baudrate, parity=serial.PARITY_NONE,
                                     stopbits=serial.STOPBITS_ONE, bytesize=serial.EIGHTBITS)
        self.commPortOpened = True

    def closeCOMMport(self, portIDtoClose):
        if (self.commPortOpened == True):
            print(">   closing COM port", portIDtoClose)
            self.ComPort.close()
        self.commPortOpened = False

    def onCloseMessageReceived(self, text):
        self.closeCOMMport(self.commPortID)

    def sendData(self):
        if self.commPortOpened == True:
            if self.ComPort.isOpen():
                sendCmd = ""
                for character in self.sendCmdString:
                    sendCmd = sendCmd + character
                    self.ComPort.write((character).encode())
            else:
                print("COM port for RK11-C Indicator Panel is not open!")
        else:
            print("COM port for RK11-C Indicator Panel is not open!")

# =================================================================================================
# =================================================================================================

    # convert a string to a numeric, if string is not a number return 100000.
    def str2int(self, s):
        ctr = i = 0
        inError = False
        for c in reversed(s):
            digit = ord(c) - 48
            if digit < 0 or digit > 9:
                inError = True
            i += digit * (10 ** ctr)
            ctr += 1
        if inError == True:
            return 100000
        else:
            return i



# == RK11-C PANEL =================================================================================

    def getStateIndex(self, cmd):
        # Based on command letter, get stateIndex
        stateIndex = 0
        if cmd == 'L':
            stateIndex = 0
        elif cmd == 'M':
            stateIndex = 1
        elif cmd == 'N':
            stateIndex = 2
        elif cmd == 'O':
            stateIndex = 3
        elif cmd == 'P':
            stateIndex = 4
        elif cmd == 'l':
            stateIndex = 5
        elif cmd == 'm':
            stateIndex = 6
        elif cmd == 'n':
            stateIndex = 7
        elif cmd == 'o':
            stateIndex = 8
        elif cmd == 'p':
            stateIndex = 9
        elif cmd == 'R':
            stateIndex = 10
        elif cmd == 'S':
            stateIndex = 11
        elif cmd == 'T':
            stateIndex = 12
        elif cmd == 'U':
            stateIndex = 13
        elif cmd == 'V':
            stateIndex = 14
        elif cmd == 'r':
            stateIndex = 15
        elif cmd == 's':
            stateIndex = 16
        elif cmd == 't':
            stateIndex = 17
        elif cmd == 'u':
            stateIndex = 18
        elif cmd == 'v':
            stateIndex = 19
        return stateIndex



    def onClickLedOn(self, event):
        number = event.GetEventObject().GetName()
        index = self.str2int(number)
        self.ledOnList[index].Hide()        # turn LED off
        self.ledOffList[index].Show()

        # update physical RK11-C panel
        termination = chr(0x0D)
        cmd = self.commandLetter[index]
        bit = self.commandBit[index]
        stateIndex = self.getStateIndex(cmd)

        currentState = self.state[stateIndex]
        newState = currentState & (255 - bit)  # turn OFF
        self.state[stateIndex] = newState

        hiHex = ( (newState & 0xF0) >> 4) + 0x30
        if hiHex > 0x39:
            hiHex = hiHex + 7  # A..F
        loHex = (newState & 0x0F) + 0x30
        if loHex > 0x39:
            loHex = loHex + 7  # A..F

        self.sendCmdString = str(cmd) + str(chr(hiHex)) + str(chr(loHex)) + str(termination)
        print ("sending command: ", self.sendCmdString)
        self.sendData()


    def onClickLedOff(self, event):
        number = event.GetEventObject().GetName()
        index = self.str2int(number)
        self.ledOffList[index].Hide()        # turn LED on
        self.ledOnList[index].Show()

        # update physical RK11-C panel
        termination = chr(0x0D)
        cmd = self.commandLetter[index]
        bit = self.commandBit[index]
        stateIndex = self.getStateIndex(cmd)

        currentState = self.state[stateIndex]
        newState = currentState | bit  # turn ON
        self.state[stateIndex] = newState

        hiHex = ( (newState & 0xF0) >> 4) + 0x30
        if hiHex > 0x39:
            hiHex = hiHex + 7  # A..F
        loHex = (newState & 0x0F) + 0x30
        if loHex > 0x39:
            loHex = loHex + 7  # A..F

        self.sendCmdString = str(cmd) + str(chr(hiHex)) + str(chr(loHex)) + str(termination)
        print ("sending command: ", self.sendCmdString)
        self.sendData()


    def paintPanel(self, evt):
        dc = wx.PaintDC(self.tabTwo)
        # draw yellow rectangle
        dc.SetPen(wx.Pen("#F4D900", 2, wx.SOLID))      # set rectangle outline yellow
        dc.SetBrush(wx.Brush("#F4D900", wx.SOLID))     # fill rectangle with yellow
        dc.DrawRectangle( 10, 10, 1138, 318 )
        # draw black rounded rectangle
        dc.SetPen(wx.Pen('black', 2, wx.SOLID))        # set rounded rectangle outline black
        dc.SetBrush(wx.Brush("#000000", wx.SOLID))     # fill rectangle with black
        dc.DrawRoundedRectangle( 19, 19, 1120, 300, 24 )
        # bar lengths above LED "groups"
        length1 = 1*30 - 11
        length2 = 2*30 - 11
        length3 = 3*30 - 11
        length4 = 4*30 - 11
        length8 = 8*30 - 11
        length16 = 16*30 - 11
        dc.SetPen(wx.Pen('#FFFFFF', 3))   # white, 3 dots thick

        # draw bars row 1
        xStart = 36
        barYpos = 49
        dc.DrawLine( xStart+0*30, barYpos, xStart+0*30 + length1, barYpos)
        dc.DrawLine( xStart+1*30, barYpos, xStart+1*30 + length1, barYpos)

        dc.DrawLine( xStart+2*30, barYpos, xStart+2*30 + length1, barYpos)
        dc.DrawLine( xStart+3*30, barYpos, xStart+3*30 + length1, barYpos)
        dc.DrawLine( xStart+4*30, barYpos, xStart+4*30 + length1, barYpos)
        dc.DrawLine( xStart+5*30, barYpos, xStart+5*30 + length1, barYpos)
        dc.DrawLine( xStart+6*30, barYpos, xStart+6*30 + length1, barYpos)
        dc.DrawLine( xStart+7*30, barYpos, xStart+7*30 + length1, barYpos)
        dc.DrawLine( xStart+8*30, barYpos, xStart+8*30 + length1, barYpos)
        dc.DrawLine( xStart+9*30, barYpos, xStart+9*30 + length1, barYpos)

        dc.DrawLine( xStart+10*30, barYpos, xStart+10*30 + length2, barYpos)
        dc.DrawLine( xStart+12*30, barYpos, xStart+12*30 + length3, barYpos)
        dc.DrawLine( xStart+15*30, barYpos, xStart+15*30 + length1, barYpos)
        dc.DrawLine( xStart+18*30, barYpos, xStart+18*30 + length16, barYpos)

        dc.DrawLine( xStart+34*30, barYpos, xStart+34*30 + length1, barYpos)
        dc.DrawLine( xStart+35*30, barYpos, xStart+35*30 + length1, barYpos)

        # draw bars row 2
        barYpos = barYpos + self.rowDistance
        dc.DrawLine( xStart+0*30, barYpos, xStart+0*30 + length3, barYpos)
        dc.DrawLine( xStart+3*30, barYpos, xStart+3*30 + length8, barYpos)

        dc.DrawLine( xStart+11*30, barYpos, xStart+11*30 + length1, barYpos)

        dc.DrawLine( xStart+12*30, barYpos, xStart+12*30 + length4, barYpos)

        dc.DrawLine( xStart+16*30, barYpos, xStart+16*30 + length1, barYpos)
        dc.DrawLine( xStart+17*30, barYpos, xStart+17*30 + length1, barYpos)

        dc.DrawLine( xStart+18*30, barYpos, xStart+18*30 + length16, barYpos)
        dc.DrawLine( xStart+35*30, barYpos, xStart+35*30 + length1, barYpos)

        # draw bars row 3
        barYpos = barYpos + self.rowDistance
        dc.DrawLine( xStart+0*30, barYpos, xStart+0*30 + length1, barYpos)
        dc.DrawLine( xStart+2*30, barYpos, xStart+2*30 + length16, barYpos)

        dc.DrawLine( xStart+18*30, barYpos, xStart+18*30 + length1, barYpos)
        dc.DrawLine( xStart+19*30, barYpos, xStart+19*30 + length1, barYpos)
        dc.DrawLine( xStart+20*30, barYpos, xStart+20*30 + length1, barYpos)
        dc.DrawLine( xStart+21*30, barYpos, xStart+21*30 + length1, barYpos)
        dc.DrawLine( xStart+22*30, barYpos, xStart+22*30 + length1, barYpos)
        dc.DrawLine( xStart+23*30, barYpos, xStart+23*30 + length1, barYpos)
        dc.DrawLine( xStart+24*30, barYpos, xStart+24*30 + length1, barYpos)
        dc.DrawLine( xStart+25*30, barYpos, xStart+25*30 + length1, barYpos)
        dc.DrawLine( xStart+26*30, barYpos, xStart+26*30 + length1, barYpos)
        dc.DrawLine( xStart+27*30, barYpos, xStart+27*30 + length1, barYpos)
        dc.DrawLine( xStart+28*30, barYpos, xStart+28*30 + length1, barYpos)
        dc.DrawLine( xStart+29*30, barYpos, xStart+29*30 + length1, barYpos)
        dc.DrawLine( xStart+30*30, barYpos, xStart+30*30 + length1, barYpos)

        dc.DrawLine( xStart+34*30, barYpos, xStart+34*30 + length1, barYpos)
        dc.DrawLine( xStart+35*30, barYpos, xStart+35*30 + length1, barYpos)

        # draw bars row 4
        barYpos = barYpos + self.rowDistance
        dc.DrawLine( xStart+0*30, barYpos, xStart+0*30 + length1, barYpos)
        dc.DrawLine( xStart+1*30, barYpos, xStart+1*30 + length1, barYpos)
        dc.DrawLine( xStart+2*30, barYpos, xStart+2*30 + length3, barYpos)

        dc.DrawLine( xStart+5*30, barYpos, xStart+5*30 + length1, barYpos)
        dc.DrawLine( xStart+6*30, barYpos, xStart+6*30 + length1, barYpos)
        dc.DrawLine( xStart+7*30, barYpos, xStart+7*30 + length1, barYpos)
        dc.DrawLine( xStart+8*30, barYpos, xStart+8*30 + length1, barYpos)
        dc.DrawLine( xStart+9*30, barYpos, xStart+9*30 + length1, barYpos)
        dc.DrawLine( xStart+10*30, barYpos, xStart+10*30 + length1, barYpos)
        dc.DrawLine( xStart+11*30, barYpos, xStart+11*30 + length1, barYpos)
        dc.DrawLine( xStart+12*30, barYpos, xStart+12*30 + length1, barYpos)
        dc.DrawLine( xStart+13*30, barYpos, xStart+13*30 + length1, barYpos)

        dc.DrawLine( xStart+14*30, barYpos, xStart+14*30 + length4, barYpos)

        dc.DrawLine( xStart+18*30, barYpos, xStart+18*30 + length1, barYpos)
        dc.DrawLine( xStart+19*30, barYpos, xStart+19*30 + length1, barYpos)
        dc.DrawLine( xStart+20*30, barYpos, xStart+20*30 + length1, barYpos)
        dc.DrawLine( xStart+21*30, barYpos, xStart+21*30 + length1, barYpos)
        dc.DrawLine( xStart+22*30, barYpos, xStart+22*30 + length1, barYpos)

        dc.DrawLine( xStart+24*30, barYpos, xStart+24*30 + length8, barYpos)
        dc.DrawLine( xStart+32*30, barYpos, xStart+32*30 + length4, barYpos)


    def ClearAllClick(self, evt):
        for row in range(4):
            for ix in range(36):
                self.ledOnList[ix+row*36].Hide()    # hide "ON" LEDs
                self.ledOffList[ix+row*36].Show()   # show"OFF" LEDs
        for ix in range(20):
            self.state[ix] = 0                      # set "shadow" state to OFF
        termination = chr(0x0D)
        cmd = "X00"                                 # send "ALL OFF" to interface
        self.sendCmdString = cmd + str(termination)
        print ("sending command: ", self.sendCmdString)
        self.sendData()



# =================================================================================================
#  TIMER
# =================================================================================================

    def startTimer(self, rate):
        if self.timerIsRunning == False:
            self.timerIsRunning = True
            self.timer.Start(rate)

    def stopTimer(self):
        if self.timerIsRunning == True:
            self.timer.Stop()
            self.timerIsRunning = False

    # handle "no COM port defined" flashing ERROR sign
    def handleCommPortError(self):
        if self.errorCommPortRate == 0:
            if self.commPortValid == False:
                # COM port not (yet) assigned
                if self.errorCommPortShown == True:
                    self.errorCommPortShown = False
                    self.errorCommPort.Hide()
                    #self.startTimer(300)
                    self.errorCommPortRate = 30
                else:
                    self.errorCommPortShown = True
                    self.errorCommPort.Show()
                    #self.startTimer(500)
                    self.errorCommPortRate = 50
            else:
                if self.errorCommPortShown == True:
                    self.errorCommPortShown = False
                    self.errorCommPort.Hide()
                #self.startTimer(200)
                self.errorCommPortRate = 30000   # no check needed anymore
        else:
            self.errorCommPortRate = self.errorCommPortRate - 1


    def OnTimer(self, event):
        self.stopTimer()
        self.handleCommPortError()              # handle "no COM port defined"
        self.startTimer(10)                     # timer granularity is 10 ms



###################################################################################################

class DIGITALFrame(wx.Frame):
    def __init__(self):
        wx.Frame.__init__(self, None, wx.ID_ANY, "DIGITAL  RK11-C  Panel", size=(1193,445))
        panel = wx.Panel(self)

        notebook = NotebookDemo(panel)
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(notebook, 1, wx.ALL|wx.EXPAND, 5)
        panel.SetSizer(sizer)
        self.Layout()
        self.Show()

#--------------------------------------------------------------------------------------------------
if __name__ == "__main__":
    app = wx.App()
    frame = DIGITALFrame()
    app.MainLoop()